<?php

/**

 * A File upgrader class for WordPress.

 *

 * This set of classes are designed to be used to upgrade/install a local set of files on the filesystem via the Filesystem Abstraction classes.

 *

 * @link http://trac.wordpress.org/ticket/7875 consolidate plugin/theme/core upgrade/install functions

 *

 * @package WordPress

 * @subpackage Upgrader

 * @since 2.8.0

 */



/**

 * WordPress Upgrader class for Upgrading/Installing a local set of files via the Filesystem Abstraction classes from a Zip file.

 *

 * @TODO More Detailed docs, for methods as well.

 *

 * @package WordPress

 * @subpackage Upgrader

 * @since 2.8.0

 */

class WTG_Upgrader {

	var $strings = array();

	var $skin = null;

	var $result = array();



	function WTG_Upgrader($skin = null) {

		return $this->__construct($skin);

	}

	function __construct($skin = null) {

		if ( null == $skin )

			$this->skin = new WTG_Upgrader_Skin();

		else

			$this->skin = $skin;

	}



	function init() {

		$this->skin->set_upgrader($this);

		$this->generic_strings();

	}



	function generic_strings() {

		$this->strings['bad_request'] = __('Invalid Data provided.');

		$this->strings['fs_unavailable'] = __('Could not access filesystem.');

		$this->strings['fs_error'] = __('Filesystem error');

		$this->strings['fs_no_root_dir'] = __('Unable to locate WordPress Root directory.');

		$this->strings['fs_no_content_dir'] = __('Unable to locate WordPress Content directory (wp-content).');

		$this->strings['fs_no_plugins_dir'] = __('Unable to locate WordPress Plugin directory.');

		$this->strings['fs_no_themes_dir'] = __('Unable to locate WordPress Theme directory.');

		$this->strings['fs_no_folder'] = __('Unable to locate needed folder (%s).');



		$this->strings['download_failed'] = __('Download failed.');

		$this->strings['installing_package'] = __('Installing the latest version.');

		$this->strings['folder_exists'] = __('Destination folder already exists.');

		$this->strings['mkdir_failed'] = __('Could not create directory.');

		$this->strings['bad_package'] = __('Incompatible Archive');



		$this->strings['maintenance_start'] = __('Enabling Maintenance mode.');

		$this->strings['maintenance_end'] = __('Disabling Maintenance mode.');

	}



	function fs_connect( $directories = array() ) {

		global $wp_filesystem;



		if ( false === ($credentials = $this->skin->request_filesystem_credentials()) )

			return false;



		if ( ! WP_Filesystem($credentials) ) {

			$error = true;

			if ( is_object($wp_filesystem) && $wp_filesystem->errors->get_error_code() )

				$error = $wp_filesystem->errors;

			$this->skin->request_filesystem_credentials($error); //Failed to connect, Error and request again

			return false;

		}



		if ( ! is_object($wp_filesystem) )

			return new WP_Error('fs_unavailable', $this->strings['fs_unavailable'] );



		if ( is_wp_error($wp_filesystem->errors) && $wp_filesystem->errors->get_error_code() )

			return new WP_Error('fs_error', $this->strings['fs_error'], $wp_filesystem->errors);



		foreach ( (array)$directories as $dir ) {

			switch ( $dir ) {

				case ABSPATH:

					if ( ! $wp_filesystem->abspath() )

						return new WP_Error('fs_no_root_dir', $this->strings['fs_no_root_dir']);

					break;

				case WP_CONTENT_DIR:

					if ( ! $wp_filesystem->wp_content_dir() )

						return new WP_Error('fs_no_content_dir', $this->strings['fs_no_content_dir']);

					break;

				case WP_PLUGIN_DIR:

					if ( ! $wp_filesystem->wp_plugins_dir() )

						return new WP_Error('fs_no_plugins_dir', $this->strings['fs_no_plugins_dir']);

					break;

				case WP_CONTENT_DIR . '/themes':

					if ( ! $wp_filesystem->find_folder(WP_CONTENT_DIR . '/themes') )

						return new WP_Error('fs_no_themes_dir', $this->strings['fs_no_themes_dir']);

					break;

				default:

					if ( ! $wp_filesystem->find_folder($dir) )

						return new WP_Error('fs_no_folder', sprintf($this->strings['fs_no_folder'], $dir));

					break;

			}

		}

		return true;

	} //end fs_connect();



	function download_package($package) {



		if ( ! preg_match('!^(http|https|ftp)://!i', $package) && file_exists($package) ) //Local file or remote?

			return $package; //must be a local file..



		if ( empty($package) )

			return new WP_Error('no_package', $this->strings['no_package']);



		$this->skin->feedback('downloading_package', $package);



		$download_file = download_url($package);



		if ( is_wp_error($download_file) )

			return new WP_Error('download_failed', $this->strings['download_failed'], $download_file->get_error_message());



		return $download_file;

	}



	function unpack_package($package, $delete_package = true) {

		global $wp_filesystem;



		$this->skin->feedback('unpack_package');



		$upgrade_folder = $wp_filesystem->wp_content_dir() . 'upgrade/';



		//Clean up contents of upgrade directory beforehand.

		$upgrade_files = $wp_filesystem->dirlist($upgrade_folder);

		if ( !empty($upgrade_files) ) {

			foreach ( $upgrade_files as $file )

				$wp_filesystem->delete($upgrade_folder . $file['name'], true);

		}



		//We need a working directory

		$working_dir = $upgrade_folder . basename($package, '.zip');



		// Clean up working directory

		if ( $wp_filesystem->is_dir($working_dir) )

			$wp_filesystem->delete($working_dir, true);



		// Unzip package to working directory

		$result = unzip_file($package, $working_dir); //TODO optimizations, Copy when Move/Rename would suffice?



		// Once extracted, delete the package if required.

		if ( $delete_package )

			unlink($package);



		if ( is_wp_error($result) ) {

			$wp_filesystem->delete($working_dir, true);

			return $result;

		}



		return $working_dir;

	}



	function install_package($args = array()) {

		global $wp_filesystem;

		$defaults = array( 'source' => '', 'destination' => '', //Please always pass these

						'clear_destination' => false, 'clear_working' => false,

						'hook_extra' => array());



		$args = wp_parse_args($args, $defaults);

		extract($args);



		@set_time_limit( 300 );



		if ( empty($source) || empty($destination) )

			return new WP_Error('bad_request', $this->strings['bad_request']);



		$this->skin->feedback('installing_package');



		$res = apply_filters('upgrader_pre_install', true, $hook_extra);

		if ( is_wp_error($res) )

			return $res;



		//Retain the Original source and destinations

		$remote_source = $source;

		$local_destination = $destination;



		$source_files = array_keys( $wp_filesystem->dirlist($remote_source) );

		$remote_destination = $wp_filesystem->find_folder($local_destination);



		//Locate which directory to copy to the new folder, This is based on the actual folder holding the files.

		if ( 1 == count($source_files) && $wp_filesystem->is_dir( trailingslashit($source) . $source_files[0] . '/') ) //Only one folder? Then we want its contents.

			$source = trailingslashit($source) . trailingslashit($source_files[0]);

		elseif ( count($source_files) == 0 )

			return new WP_Error('bad_package', $this->strings['bad_package']); //There are no files?

		//else //Its only a single file, The upgrader will use the foldername of this file as the destination folder. foldername is based on zip filename.



		//Hook ability to change the source file location..

		$source = apply_filters('upgrader_source_selection', $source, $remote_source, $this);

		if ( is_wp_error($source) )

			return $source;



		//Has the source location changed? If so, we need a new source_files list.

		if ( $source !== $remote_source )

			$source_files = array_keys( $wp_filesystem->dirlist($source) );



		//Protection against deleting files in any important base directories.

		if ( in_array( $destination, array(ABSPATH, WP_CONTENT_DIR, WP_PLUGIN_DIR, WP_CONTENT_DIR . '/themes') ) ) {

			$remote_destination = trailingslashit($remote_destination) . trailingslashit(basename($source));

			$destination = trailingslashit($destination) . trailingslashit(basename($source));

		}



		if ( $wp_filesystem->exists($remote_destination) ) {

			if ( $clear_destination ) {

				//We're going to clear the destination if theres something there

				$this->skin->feedback('remove_old');

				$removed = $wp_filesystem->delete($remote_destination, true);

				$removed = apply_filters('upgrader_clear_destination', $removed, $local_destination, $remote_destination, $hook_extra);



				if ( is_wp_error($removed) )

					return $removed;

				else if ( ! $removed )

					return new WP_Error('remove_old_failed', $this->strings['remove_old_failed']);

			} else {

				//If we're not clearing the destination folder and something exists there allready, Bail.

				//But first check to see if there are actually any files in the folder.

				$_files = $wp_filesystem->dirlist($remote_destination);

				if ( ! empty($_files) ) {

					$wp_filesystem->delete($remote_source, true); //Clear out the source files.

					return new WP_Error('folder_exists', $this->strings['folder_exists'], $remote_destination );

				}

			}

		}



		//Create destination if needed

		if ( !$wp_filesystem->exists($remote_destination) )

			if ( !$wp_filesystem->mkdir($remote_destination, FS_CHMOD_DIR) )

				return new WP_Error('mkdir_failed', $this->strings['mkdir_failed'], $remote_destination);



		// Copy new version of item into place.

		$result = copy_dir($source, $remote_destination);

		if ( is_wp_error($result) ) {

			if ( $clear_working )

				$wp_filesystem->delete($remote_source, true);

			return $result;

		}



		//Clear the Working folder?

		if ( $clear_working )

			$wp_filesystem->delete($remote_source, true);



		$destination_name = basename( str_replace($local_destination, '', $destination) );

		if ( '.' == $destination_name )

			$destination_name = '';



		$this->result = compact('local_source', 'source', 'source_name', 'source_files', 'destination', 'destination_name', 'local_destination', 'remote_destination', 'clear_destination', 'delete_source_dir');



		$res = apply_filters('upgrader_post_install', true, $hook_extra, $this->result);

		if ( is_wp_error($res) ) {

			$this->result = $res;

			return $res;

		}



		//Bombard the calling function will all the info which we've just used.

		return $this->result;

	}



	function run($options) {



		$defaults = array( 	'package' => '', //Please always pass this.

							'destination' => '', //And this

							'clear_destination' => false,

							'clear_working' => true,

							'is_multi' => false,

							'hook_extra' => array() //Pass any extra $hook_extra args here, this will be passed to any hooked filters.

						);



		$options = wp_parse_args($options, $defaults);

		extract($options);



		//Connect to the Filesystem first.

		$res = $this->fs_connect( array(WP_CONTENT_DIR, $destination) );

		if ( ! $res ) //Mainly for non-connected filesystem.

			return false;



		if ( is_wp_error($res) ) {

			$this->skin->error($res);

			return $res;

		}



		if ( !$is_multi ) // call $this->header separately if running multiple times

			$this->skin->header();



		$this->skin->before();



		//Download the package (Note, This just returns the filename of the file if the package is a local file)

		$download = $this->download_package( $package );

		if ( is_wp_error($download) ) {

			$this->skin->error($download);

			return $download;

		}



		//Unzip's the file into a temporary directory

		$working_dir = $this->unpack_package( $download );

		if ( is_wp_error($working_dir) ) {

			$this->skin->error($working_dir);

			return $working_dir;

		}



		//With the given options, this installs it to the destination directory.

		$result = $this->install_package( array(

											'source' => $working_dir,

											'destination' => $destination,

											'clear_destination' => $clear_destination,

											'clear_working' => $clear_working,

											'hook_extra' => $hook_extra

										) );

		$this->skin->set_result($result);

		if ( is_wp_error($result) ) {

			$this->skin->error($result);

			$this->skin->feedback('process_failed');

		} else {

			//Install Suceeded

			$this->skin->feedback('process_success');

		}

		$this->skin->after();



		if ( !$is_multi )

			$this->skin->footer();



		return $result;

	}



	function maintenance_mode($enable = false) {

		global $wp_filesystem;

		$file = $wp_filesystem->abspath() . '.maintenance';

		if ( $enable ) {

			$this->skin->feedback('maintenance_start');

			// Create maintenance file to signal that we are upgrading

			$maintenance_string = '<?php $upgrading = ' . time() . '; ?>';

			$wp_filesystem->delete($file);

			$wp_filesystem->put_contents($file, $maintenance_string, FS_CHMOD_FILE);

		} else if ( !$enable && $wp_filesystem->exists($file) ) {

			$this->skin->feedback('maintenance_end');

			$wp_filesystem->delete($file);

		}

	}



}



/**

 * Plugin Upgrader class for WordPress Plugins, It is designed to upgrade/install plugins from a local zip, remote zip URL, or uploaded zip file.

 *

 * @TODO More Detailed docs, for methods as well.

 *

 * @package WordPress

 * @subpackage Upgrader

 * @since 2.8.0

 */

class Plugin_Upgrader extends WTG_Upgrader {



	var $result;

	var $bulk = false;

	var $show_before = '';



	function upgrade_strings() {

		$this->strings['up_to_date'] = __('The plugin is at the latest version.');

		$this->strings['no_package'] = __('Upgrade package not available.');

		$this->strings['downloading_package'] = __('Downloading update.');

		$this->strings['unpack_package'] = __('Unpacking the update.');

		$this->strings['deactivate_plugin'] = __('Deactivating the plugin.');

		$this->strings['remove_old'] = __('Removing the old version of the plugin.');

		$this->strings['remove_old_failed'] = __('Could not remove the old plugin.');

		$this->strings['process_failed'] = __('Plugin upgrade Failed.');

		$this->strings['process_success'] = __('Plugin upgraded successfully.');

	}



	function install_strings() {

		$this->strings['no_package'] = __('Install package not available.');

		$this->strings['downloading_package'] = __('Downloading install package.');

		$this->strings['unpack_package'] = __('Unpacking the package.');

		$this->strings['installing_package'] = __('Installing the plugin.');

		$this->strings['process_failed'] = __('Plugin Install Failed.');

		$this->strings['process_success'] = __('Plugin Installed successfully.');

	}



	function upgrade($plugin, $package) {



		$this->init();

		$this->upgrade_strings();



		add_filter('upgrader_pre_install', array(&$this, 'deactivate_plugin_before_upgrade'), 10, 2);

		add_filter('upgrader_clear_destination', array(&$this, 'delete_old_plugin'), 10, 4);

		//'source_selection' => array(&$this, 'source_selection'), //theres a track ticket to move up the directory for zip's which are made a bit differently, useful for non-.org plugins.



		$this->run(array(

					'package' => $package,

					'destination' => WP_PLUGIN_DIR,

					'clear_destination' => true,

					'clear_working' => true,

					'hook_extra' => array(

								'plugin' => $plugin

					)

				));



		// Cleanup our hooks, incase something else does a upgrade on this connection.

		remove_filter('upgrader_pre_install', array(&$this, 'deactivate_plugin_before_upgrade'));

		remove_filter('upgrader_clear_destination', array(&$this, 'delete_old_plugin'));



		if ( ! $this->result || is_wp_error($this->result) )

			return $this->result;



		// Force refresh of plugin update information

		delete_transient('update_plugins');

	}



	//return plugin info.

	function plugin_info() {

		if ( ! is_array($this->result) )

			return false;

		if ( empty($this->result['destination_name']) )

			return false;



		$plugin = get_plugins('/' . $this->result['destination_name']); //Ensure to pass with leading slash

		if ( empty($plugin) )

			return false;



		$pluginfiles = array_keys($plugin); //Assume the requested plugin is the first in the list



		return $this->result['destination_name'] . '/' . $pluginfiles[0];

	}

	

	//Hooked to pre_install

	function deactivate_plugin_before_upgrade($return, $plugin) {



		if ( is_wp_error($return) ) //Bypass.

			return $return;



		$plugin = isset($plugin['plugin']) ? $plugin['plugin'] : '';

		if ( empty($plugin) )

			return new WP_Error('bad_request', $this->strings['bad_request']);



		if ( is_plugin_active($plugin) ) {

			$this->skin->feedback('deactivate_plugin');

			//Deactivate the plugin silently, Prevent deactivation hooks from running.

			deactivate_plugins($plugin, true);

		}

	}



	//Hooked to upgrade_clear_destination

	function delete_old_plugin($removed, $local_destination, $remote_destination, $plugin) {

		global $wp_filesystem;



		if ( is_wp_error($removed) )

			return $removed; //Pass errors through.



		$plugin = isset($plugin['plugin']) ? $plugin['plugin'] : '';

		if ( empty($plugin) )

			return new WP_Error('bad_request', $this->strings['bad_request']);



		$plugins_dir = $wp_filesystem->wp_plugins_dir();

		$this_plugin_dir = trailingslashit( dirname($plugins_dir . $plugin) );



		if ( ! $wp_filesystem->exists($this_plugin_dir) ) //If its already vanished.

			return $removed;



		// If plugin is in its own directory, recursively delete the directory.

		if ( strpos($plugin, '/') && $this_plugin_dir != $plugins_dir ) //base check on if plugin includes directory seperator AND that its not the root plugin folder

			$deleted = $wp_filesystem->delete($this_plugin_dir, true);

		else

			$deleted = $wp_filesystem->delete($plugins_dir . $plugin);



		if ( ! $deleted )

			return new WP_Error('remove_old_failed', $this->strings['remove_old_failed']);



		return $removed;

	}

}





/**

 * Generic Skin for the WordPress Upgrader classes. This skin is designed to be extended for specific purposes.

 *

 * @TODO More Detailed docs, for methods as well.

 *

 * @package WordPress

 * @subpackage Upgrader

 * @since 2.8.0

 */

class WTG_Upgrader_Skin {



	var $upgrader;

	var $done_header = false;



	function WTG_Upgrader_Skin($args = array()) {

		return $this->__construct($args);

	}

	function __construct($args = array()) {

		$defaults = array( 'url' => '', 'nonce' => '', 'title' => '', 'context' => false );

		$this->options = wp_parse_args($args, $defaults);

	}



	function set_upgrader(&$upgrader) {

		if ( is_object($upgrader) )

			$this->upgrader =& $upgrader;

	}

	function set_result($result) {

		$this->result = $result;

	}



	function request_filesystem_credentials($error = false) {

		$url = $this->options['url'];

		$context = $this->options['context'];

		if ( !empty($this->options['nonce']) )

			$url = wp_nonce_url($url, $this->options['nonce']);

		return request_filesystem_credentials($url, '', $error, $context); //Possible to bring inline, Leaving as is for now.

	}



	function header() {

		if ( $this->done_header )

			return;

		$this->done_header = true;

		echo '<div class="wrap">';

		echo screen_icon();

		echo '<h2>' . $this->options['title'] . '</h2>';

	}

	function footer() {

		echo '</div>';

	}



	function error($errors) {

		if ( ! $this->done_header )

			$this->header();

		if ( is_string($errors) ) {

			$this->feedback($errors);

		} elseif ( is_wp_error($errors) && $errors->get_error_code() ) {

			foreach ( $errors->get_error_messages() as $message ) {

				if ( $errors->get_error_data() )

					$this->feedback($message . ' ' . $errors->get_error_data() );

				else

					$this->feedback($message);

			}

		}

	}



	function feedback($string) {

		if ( isset( $this->upgrader->strings[$string] ) )

			$string = $this->upgrader->strings[$string];



		if ( strpos($string, '%') !== false ) {

			$args = func_get_args();

			$args = array_splice($args, 1);

			if ( !empty($args) )

				$string = vsprintf($string, $args);

		}

		if ( empty($string) )

			return;

		show_message($string);

	}

	function before() {}

	function after() {}



}



/**

 * Plugin Upgrader Skin for WordPress Plugin Upgrades.

 *

 * @TODO More Detailed docs, for methods as well.

 *

 * @package WordPress

 * @subpackage Upgrader

 * @since 2.8.0

 */

class Plugin_Upgrader_Skin extends WTG_Upgrader_Skin {

	var $plugin = '';

	var $plugin_active = false;



	function Plugin_Upgrader_Skin($args = array()) {

		return $this->__construct($args);

	}



	function __construct($args = array()) {

		$defaults = array( 'url' => '', 'plugin' => '', 'nonce' => '', 'title' => __('Upgrade Plugin') );

		$args = wp_parse_args($args, $defaults);



		$this->plugin = $args['plugin'];



		$this->plugin_active = is_plugin_active($this->plugin);



		parent::__construct($args);

	}



	function after() {

		if ( $this->upgrader->bulk )

			return;



		$this->plugin = $this->upgrader->plugin_info();

		if( !empty($this->plugin) && !is_wp_error($this->result) && $this->plugin_active ){

			show_message(__('Attempting reactivation of the plugin'));

			echo '<iframe style="border:0;overflow:hidden" width="100%" height="170px" src="' . wp_nonce_url('update.php?action=activate-plugin&plugin=' . $this->plugin, 'activate-plugin_' . $this->plugin) .'"></iframe>';

		}



		$update_actions =  array(

			'activate_plugin' => '<a href="' . wp_nonce_url('plugins.php?action=activate&amp;plugin=' . $this->plugin, 'activate-plugin_' . $this->plugin) . '" title="' . esc_attr__('Activate this plugin') . '" target="_parent">' . __('Activate Plugin') . '</a>',

			'plugins_page' => '<a href="' . admin_url('plugins.php') . '" title="' . esc_attr__('Goto plugins page') . '" target="_parent">' . __('Return to Plugins page') . '</a>'

		);

		if ( $this->plugin_active )

			unset( $update_actions['activate_plugin'] );

		if ( ! $this->result || is_wp_error($this->result) )

			unset( $update_actions['activate_plugin'] );



		$update_actions = apply_filters('update_plugin_complete_actions', $update_actions, $this->plugin);

		if ( ! empty($update_actions) )

			$this->feedback('<strong>' . __('Actions:') . '</strong> ' . implode(' | ', (array)$update_actions));

	}



	function before() {

		if ( $this->upgrader->show_before ) {

			echo $this->upgrader->show_before;

			$this->upgrader->show_before = '';

		}

	}

}



